package gov.va.genisis2.controller;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import gov.va.genisis2.bo.BusinessService;
import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.exceptions.ErrorResponse;
import gov.va.genisis2.exceptions.Genisis2RestException;
import gov.va.genisis2.model.CommentHistory;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;
import gov.va.genisis2.vo.CopyTableSource;

/**
 * The Class Genisis2Controller.
 */
@CrossOrigin(origins = "*")
@RestController
@RequestMapping("/services")
public class Genisis2Controller {
	
	/** The logger. */
	private static final Logger LOGGER = LoggerFactory.getLogger(Genisis2Controller.class);

	/** The business service. */
	private BusinessService businessService;

	
	/* T A B L E  C O P Y   T R A N S F E R    R E S T     C A L L S */
	 
	/**
	 * 
	 * Performs the table copy script execution.
	 * 
	 * @param requestId the request id of the data to be transferred
	 * @param copyTableSource the source table name 
	 * @return a json response
	 * @throws Genisis2RestException Genisis2RestException
	 */
	@RequestMapping(value = "/requests/{requestId}/copytable", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity performTableCopyTransfer(@PathVariable int requestId, @RequestBody CopyTableSource copyTableSource) throws Genisis2RestException {
		LOGGER.info("Table Copy Transfer.");
		ResponseWrapper wrapper = businessService.performTableCopyTransfer(copyTableSource, requestId);
		return createReposneWrapper(wrapper);
	}

	
	/**
	 * @return a json response
	 * @throws Genisis2RestException Genisis2RestException
	 */
	@RequestMapping(value = "/copytable/domains", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getCopyTableDomains() throws Genisis2RestException {
		LOGGER.info("Copy Table Domains.");
		ResponseWrapper wrapper = businessService.getCopyTableDomains();
		return createReposneWrapper(wrapper);
	}
	
	/* U S E R    M A N A G E ME N T     R E S T     C A L L S */
	
	/**
	 * Gets the user details by email.
	 *
	 * @param email the email
	 * @return a json response
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/users/{email}/email", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getUserDetailsByEmail(@PathVariable String email) throws Genisis2RestException {
		LOGGER.info(" Get user by email.");
		ResponseWrapper wrapper = businessService.getUserDetailsByEmail(email);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the user details by id.
	 *
	 * @param id the id
	 * @return the user details by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/users/{id:.+}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getUserDetailsById(@PathVariable int id) throws Genisis2RestException {
		LOGGER.info(" Get user by Id.");
		ResponseWrapper wrapper = businessService.getUserDetailsById(id);
		return createReposneWrapper(wrapper);
	}	

			
	/**
	 * Gets the user role by uid.
	 *
	 * @param uid the uid
	 * @return the user role by uid
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/users/role/{uid}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getUserRoleByUid(@PathVariable int uid) throws Genisis2RestException {
		LOGGER.info(" Get user role by uid.");
		ResponseWrapper wrapper = businessService.getUserRole(uid);
		return createReposneWrapper(wrapper);
	}
	
	/* R E Q U E S T       R E S T      C A L L S */
	/**
	 * Creates the request.
	 *
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests", method = RequestMethod.POST, headers = "Accept=*/*", consumes = "application/json", produces = "application/json")
	public ResponseEntity createRequest(@RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Create Request ");
		ResponseWrapper wrapper = businessService.createRequest(request);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the all requests by study approval.
	 *
	 * @param studyApprovalId the study approval id
	 * @return the all requests by study approval
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{studyApprovalId}/studyApproval", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequestsByStudyApproval(@PathVariable int studyApprovalId)
			throws Genisis2RestException {
		LOGGER.info("getAllRequestsByStudyApproval ");
		ResponseWrapper wrapper = businessService.getAllRequestsByStudyApproval(studyApprovalId);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the requests by id.
	 *
	 * @param id the id
	 * @return the requests by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsById(@PathVariable int id) throws Genisis2RestException {
		LOGGER.info("Get Request  by Id ");
		ResponseWrapper wrapper = businessService.getRequestByID(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the requests by user.
	 *
	 * @param uid the uid
	 * @return the requests by user
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{uid}/createdBy", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestsByUser(@PathVariable String uid) throws Genisis2RestException {
		LOGGER.info("Get Requests  by user ");
		ResponseWrapper wrapper = businessService.getRequestsByUID(uid);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all requests.
	 *
	 * @return the all requests
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllRequests() throws Genisis2RestException {
		LOGGER.info("Get All Requests ");
		ResponseWrapper wrapper = businessService.getAllRequests();
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by data manager.
	 *
	 * @param id the id
	 * @return the request by data manager
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/dataManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByDataManager(@PathVariable String id) throws Genisis2RestException {
		LOGGER.info("Request by DataManager  ");
		ResponseWrapper wrapper = businessService.getRequestsByDataManagers(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by data source manager.
	 *
	 * @param id the id
	 * @return the request by data manager
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/dataSourceManager", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByDataSourceManager(@PathVariable String id) throws Genisis2RestException {
		LOGGER.info("Request by Data Source Manager  ");
		ResponseWrapper wrapper = businessService.getRequestsDataSourceManagers(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the request by status.
	 *
	 * @param status the status
	 * @return the request by status
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{status}/status", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestByStatus(@PathVariable String status) throws Genisis2RestException {
		LOGGER.info("Requests by Status /approved/status");
		ResponseWrapper wrapper = businessService.getAllRequestsByStaus(status);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Request Update");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_UPDATE.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Submit request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/submit", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity submitRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Request Submit ");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_SUBMIT.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Sent request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/sent", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity sentRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Request Sent/Approved ");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_SENT.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Deny request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/deny", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity denyRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Request Deny  ");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_DENIED.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Return request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/return", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity returnRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info(" Request Returned  ");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_RETURNED.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Fulfill request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/acceptdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity fulfillRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Accept Data");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_ACCEPTED.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Un fulfill request.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/denydata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity unFulfillRequest(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Reject Data");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.REQUEST_NOT_ACCEPTED.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Reject data.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/rejectdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity rejectData(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Reject Results");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.RESULTS_NOT_ACCEPTED.getText());
		return createReposneWrapper(wrapper);
	}

	/**
	 * Accept data.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/confirmdata", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity acceptData(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Accept Results");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.RESULTS_ACCEPTED.getText());
		return createReposneWrapper(wrapper);
	}
		
	/**
	 * Request Deliver.
	 *
	 * @param id the id
	 * @param request the request
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requests/{id}/resultsdelivered", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity requestDeliver(@PathVariable int id, @RequestBody Request request) throws Genisis2RestException {
		LOGGER.info("Results Delivered");
		ResponseWrapper wrapper = businessService.persist(id, request, CommonEnum.RESULTS_DELIVERED.getText());
		return createReposneWrapper(wrapper);
	}
	
	/* S T U D Y    A P P R O V A L     R E S T     C A L L S */

	/**
	 * Creates the study approval.
	 *
	 * @param studyApproval the study approval
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity createStudyApproval(@RequestBody StudyApproval studyApproval) throws Genisis2RestException {
		LOGGER.info("Create Study Approval");
		ResponseWrapper wrapper = businessService.createStudyApproval(studyApproval);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Update study approval.
	 *
	 * @param id the id
	 * @param studyApproval the study approval
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.PUT, consumes = "application/json", produces = "application/json")
	public ResponseEntity updateStudyApproval(@PathVariable int id, @RequestBody StudyApproval studyApproval) throws Genisis2RestException {
		if (id > 0) {
			studyApproval.setId(id);
			ResponseWrapper wrapper = businessService.updateStudyApproval(studyApproval);
			LOGGER.info("Updated Study Approval");
			return createReposneWrapper(wrapper);
		} else {
			ResponseWrapper wrapper = new ResponseWrapper();
			LOGGER.info("No ID Found");
			wrapper.setSuccess(false);
			wrapper.setMessage("No ID Found");
			return new ResponseEntity<ResponseWrapper>(wrapper, HttpStatus.NOT_MODIFIED);
		}
	}

	/**
	 * Gets the study approval by id.
	 *
	 * @param id the id
	 * @return the study approval by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{id}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalById(@PathVariable int id) throws Genisis2RestException {
		LOGGER.info("Study Approval by Id");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByID(id);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the study approval by user.
	 *
	 * @param uid the uid
	 * @return the study approval by user
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals/{uid}/{user}", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getStudyApprovalByUser(@PathVariable String uid) throws Genisis2RestException {
		LOGGER.info("Study Approval by User");
		ResponseWrapper wrapper = businessService.getStudyApprovalsByUID(uid);
		return createReposneWrapper(wrapper);
	}

	/**
	 * Gets the all study approvals.
	 *
	 * @return the all study approvals
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/studyApprovals", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getAllStudyApprovals() throws Genisis2RestException {
		LOGGER.info("All Study Approvals");
		ResponseWrapper wrapper = businessService.getStudyApprovals();
		return createReposneWrapper(wrapper);
	}
	
	/* C O M M E N T    H I S T O R Y     R E S T    C A L L S */
	
	/**
	 * Creates the comment history.
	 *
	 * @param commentHistory the comment history
	 * @return the response entity
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/comment", method = RequestMethod.POST, consumes = "application/json", produces = "application/json")
	public ResponseEntity createCommentHistory(@RequestBody CommentHistory commentHistory) throws Genisis2RestException {
		LOGGER.info("Create Comment History");
		ResponseWrapper wrapper = businessService.createCommentHistory(commentHistory);
		return createReposneWrapper(wrapper);
	}

	/* ACTIVITI R E S T     C A L L S */
	/**
	 * Gets the request tracking by id.
	 *
	 * @param id the id
	 * @return the request tracking by id
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	// Request History 
	@RequestMapping(value = "/requests/{id}/log", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestTrackingById(@PathVariable int id) throws Genisis2RestException {
		LOGGER.info("Get Request Tracking by Id ");
		ResponseWrapper wrapper = businessService.getRequestTrackingByID(id);
		return createReposneWrapper(wrapper);
	}
	
	/**
	 * Gets the request data sources.
	 *
	 * @return the request data sources
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/datasources", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestDataSources() throws Genisis2RestException {
		LOGGER.info("All Request Data Sources");
		ResponseWrapper wrapper = businessService.getRequestDataSources();
		return createReposneWrapper(wrapper);
	}
		
	/**
	 * Gets the request types.
	 *
	 * @return the request types
	 * @throws Genisis2RestException the genisis 2 rest exception
	 */
	@RequestMapping(value = "/requestTypes", method = RequestMethod.GET, consumes = "application/json", produces = "application/json")
	public ResponseEntity getRequestTypes() throws Genisis2RestException {
		LOGGER.info("All Request Types");
		ResponseWrapper wrapper = businessService.getRequestTypes();
		return createReposneWrapper(wrapper);
	}
		
	/**
	 * Creates the reposne wrapper.
	 *
	 * @param wrapper the wrapper
	 * @return the response entity
	 */
	private ResponseEntity<ResponseWrapper> createReposneWrapper(ResponseWrapper wrapper) {
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Pragma", "No-cache");
		responseHeaders.add("Cache-Control", "no-cache");
		return new ResponseEntity<>(wrapper, responseHeaders, HttpStatus.OK);
	}
	
	/**
	 * Exception handler.
	 *
	 * @param ex the ex
	 * @return the response entity
	 */
	@ExceptionHandler(Genisis2RestException.class)
	public ResponseEntity<ErrorResponse> exceptionHandler(Exception ex) {
		ErrorResponse error = new ErrorResponse();
		error.setErrorCode(HttpStatus.PRECONDITION_FAILED.value());
		error.setMessage(ex.getMessage());
		return new ResponseEntity<>(error, HttpStatus.OK);
	}
	
	/**
	 * Gets the business service.
	 *
	 * @return the business service
	 */
	public BusinessService getBusinessService() {
		return businessService;
	}

	/**
	 * Sets the business service.
	 *
	 * @param businessService the new business service
	 */
	@Autowired
	public void setBusinessService(BusinessService businessService) {
		this.businessService = businessService;
	}
}
